#!/bin/sh
# LICENSE

# FOR CONSTANTS, EDIT ./qownconstants.inc


# FUNCTIONS AND PROCEDURES

writelog(){
   local timestampp=`date +"%Y/%m/%d_%H:%M:%S"`
   [ "$#" = "0" ] || echo -n "$timestampp " >> $qb_log_file
   echo "$@" >> $qb_log_file
}

abort_because_wrongdir(){
   currdir=`pwd`
   echo ""
   echo "FATAL ERROR: You tried to run $0 from the wrong directory\n  ($currdir)."
   echo '  There is no qownconstants.inc in this directory.'
   echo '  Run this program from the directory with qownconstants.inc.'
   echo ""
   exit 1
}

usage_and_exit(){
   echo ""
   [ "$1" = "" ] || echo "ERROR: $1"  
   echo ''
   echo 'USAGE: qownbackup.sh ['
   echo '   backup'
   echo ' | sync'
   echo ' | die'
   echo ' | list'
   echo ' | check'
   echo ' | versions <filename>'
   echo ' | versions_long <filename> (not implemented yet)'
   echo ']'
   echo ''
   echo '* backup starts a continuous backup of .md files in qownNotes dir.'
   echo '* sync copies all of latest dir to an incremental dir.'
   echo '* die kills all qownbackup loop procs for this user.'
   echo '* list lists all latest backed up Markdown files for this user.'
   echo '* check lists all ./qownbackup.sh backup processes for this user.'
   echo '* versions lists all versions of <filename>.'
   echo '* versions_long lists all versions of <filename>, with contents.'
   echo '1234567890123456789012345678901234567890123456789012345678901234567890'
   echo ''
   exit 1
}


check_for_existing_loop_and_exit(){
   ps -U $USER -o ppid,pid,user,command | head -n1
   ps -U $USER -o ppid,pid,user,command | grep "\./qownbackup.sh\s\s*backup"
   exit 0
}


sync_(){
   local qb_incdir=`date +$qb_qown_backup_dir/inc%Y/%m/%d/%H/%M`
   mkdir -p $qb_incdir
   writelog
   writelog "Sync function called, creating $qb_incdir before copying all .md files."
   2>&1 cp -pv $qb_qown_backup_latest_dir/*.md  $qb_incdir >> $qb_log_file
   writelog
}

kill_backup_loops(){
   ps -U $USER
}

cp_faux(){  # for diagnosis only
   echo "cp -p  $@"
}

mkdir_faux(){  # for diagnosis only
   echo "mkdir   $1   $2"
}

check1file(){
   local pathname=$1
   local filename=`basename "$pathname"`
   local dstname="$qb_qown_backup_latest_dir/$filename"
   [ ! -f $distname ] && return 0
   if diff -q "$pathname" "$dstname" > /dev/null; then
      return 1
   else
      return 0
   fi
}

abort_because_not_implemented(){
   echo "Command $1 not implemented in qownbackup at this time."
   exit 1
}

abort_because_bad_user(){
   echo "ERROR: qownbackup arg 1, $1, doesn\'t match user $USER."
   exit 1
}


abort_because_loop_already_running(){
   echo "\nERROR: Tried to run second backup loop for user $USER.\n"
   exit 1
}

# DANGER!!!!!
# THE abort_if_backup_for_this_user_already_running FUNCTION
# LOOKS LIKE THERE ARE A THOUSAND BETTER WAYS TO DO IT, BUT
# ALL THE OTHER WAYS I TRIED FAILED.
# I WAS UNABLE TO FIND ANOTHER WAY THAT WORKED CORRECTLY.
# DO NOT CHANGE THIS FUNCTION FOR STYLE OR ANY OTHER NON-VITAL
# REASON. IF YOU ABSOLUTELY MUST CHANGE IT,
# BACK UP THIS FILE BEFORE CHANGING THIS FUNCTION.
# YOU HAVE BEEN WARNED.

abort_if_backup_for_this_user_already_running(){
   local tempfile=`mktemp --tmpdir qownprocs.XXXXXXXX`
   echo "." > $tempfile
   ps -U $USER -o command | grep "qownbackup.sh\s\s*backup" | wc -l > $tempfile
   local resultt=`cat $tempfile`
   rm $tempfile
   #echo $$   # diagnostic
   #echo "resultt is $resultt"      # diagnostic
   [ "$resultt" = "1" ] || abort_because_loop_already_running
}





run_backup_loop(){
   writelog 
   writelog "Starting qownbackup system backup loop, process $$"

   # OUTER LOOP
   while true; do
      local qb_incdir=`date +$qb_qown_backup_dir/inc%Y/%m/%d/%H/%M`
      #writelog "QownBackupLoop creating directory $qb_incdir"
      #echo $qb_incdir
      qb_files="$qb_qown_dir/*.md"
      qb_mod_flag=0
   
      # INNER LOOP
      # writelog "."
      for f in $qb_files; do
         if check1file "$f"; then
            qb_mod_flag=1
   	 qb_basename=`basename "$f"`
   	 cp -p "$f" "$qb_qown_backup_latest_dir/$qb_basename"
	 writelog "Created backup file $qb_qown_backup_latest_dir/$qb_basename"
         mkdir -p $qb_incdir
   	 cp -p "$qb_qown_backup_latest_dir/$qb_basename" "$qb_incdir/$qb_basename"
   	 writelog "Created backup file $qb_incdir/$qb_basename"
         fi
      done
      [ "$qb_mod_flag" = "1" ] && writelog
      sleep $qb_interval_seconds
   done
   writelog
   writelog "Qownbackup system backup loop has died or been killed."
}


print_versions_old(){
   echo ""
   find $qb_qown_backup_dir | grep "$1" | sort -r
   echo ""
}

print1version(){
   stat --printf "%10s\t%n\n" $1
}

print_versions(){
   local fname
   echo ""
   find $qb_qown_backup_dir | grep "$2" | sort -r \
	   | while read fname; do
           print1version "$fname";
   done
   echo ""
}

list_latest(){
   ls -1 "$qb_qown_backup_latest_dir"
}

# WARNING: MANY WRONG WAYS TO DO THE FOLLOWING:
kill_all_backup_loops(){
   local pid
   local cmd
   ps x -o pid,command | grep "./qownbackup.sh\s\s*backup" \
      | while read pid cmd; do
         writelog "Killing backup process $pid, command=$cmd"
         kill $pid
      done
}

handle_command_line(){
   [ "$#" = "0" ] && usage_and_exit ""
   echo $1 | grep "^versions" > /dev/null && [ $# -lt 2 ] && usage_and_exit "Wrong number of arguments ($#)"
   echo $1 | grep "^versions" > /dev/null && [ $# -gt 1 ] && print_versions "$@" && exit 0
   [ "$#" != "1" ] && usage_and_exit "Wrong number of arguments ($#)"
   [ "$1" = "sync"     ] && sync_ && exit 0
   [ "$1" = "list"     ] && list_latest && exit 0
   [ "$1" = "die"      ] && kill_all_backup_loops && exit 0
   [ "$1" = "versions_long" ] && abort_because_not_implemented versions_long && exit 0
   [ "$1" = "check" ] && check_for_existing_loop_and_exit && exit 0

   # This leaves backup as the only remaining legal command
   if test "$1" = "backup"; then
      # [ "$username" = "$USER" ] || baduser $username
      abort_if_backup_for_this_user_already_running $@
      run_backup_loop $@
      return 0
   else
      usage_and_exit "Wrong command ($1)"
   fi
}


create_directories_if_necessary(){
   mkdir -p $qb_qown_dir
   mkdir -p $qb_qown_backup_dir
   mkdir -p $qb_qown_backup_latest_dir
   mkdir -p $qb_log_dir
}


# MAIN PROCEDURE
main(){
   [ -f ./qownconstants.inc ] || abort_wrongdir
   . ./qownconstants.inc
   create_directories_if_necessary
   handle_command_line $@ || exit 1
}

main $@
